/*=================================================================
 *
 * cond_loss_dist_c2.c ... Compute the conditional portfolio loss distribution (conditional on common risk factors Z) of a
 *                        portfolio of credits using the recursive method by Andersen, Sidenius and Basu.
 *
 * The calling syntax is:
 *      cond_loss_dist_c2(ai_row_vector, p_surv_idio_row_vector, int_common_factor)
 *
 * Sample call:
 *      cond_loss_dist_c2(ones(1,10), repmat(0.9, 1, 10), 0)
 *      cond_loss_dist_c2(ones(1,10), repmat(1, 1, 10), -log(0.9))
 *
 *
 *=================================================================*/

#include <math.h>
#include "mex.h"

#ifndef max
#define max(a,b) (((a) > (b)) ? (a) : (b))
#endif

/* Recursive computation of loss distribution */
static void cond_loss_c2(double p[], double loss[], unsigned int num_firms)
{
    int i, j;
    double exp_def = 0;
    int threshold;
    
    // Compute expected number of defaults
    for (i=0; i<num_firms; i++) 
    {
        exp_def += p[i];
    }
     
    // Calculate threshold K, which bounds the number of defaults with prob. 1-1e-14
    threshold = (int) (8 + exp_def + 8*sqrt(exp_def));
    threshold = -max(-num_firms, - threshold);
    
    loss[0] = 1;
    for (i=0; i<num_firms; i++) 
    {
        // Clear memory from potentially previous run of function
        loss[i+1] = 0;

        // Update loss distribution when adding one asset
        for (j=-max(-i-1,-threshold); j>=1; j--)
        {
            loss[j] = loss[j]*(1-p[i]) + loss[j-1]*p[i];
        }
        loss[0] = loss[0] * (1-p[i]);
    }
    //loss[0] = (double) threshold;
    return;
}


// Wrapper function around cond_loss_dist, that merges idiosyncratic and systematic survival probabilities in order to calculate default probabilities
static void cond_loss_dist_wrapper(double loss[], double ai[], double p_surv_idio[], double Y_int[], unsigned int num_firms)
{
    int i;
    double *p_def;
    
    // Allocate memory for temporary variables
    p_def = mxGetPr(mxCreateDoubleMatrix(num_firms, 1, mxREAL)); 
    
    // Calculate conditional default probabilities
    for (i=0; i<num_firms; i++)
    {
        p_def[i] = 1 - p_surv_idio[i] * exp( -Y_int[0] * ai[i]);
    }

    // Calculate conditional portfolio loss distribution
    cond_loss_c2(p_def, loss, num_firms);
    return;
}

/* Gateway routine (to Matlab) */
/* sample call: horizon_loss_dist_c2([1; 2], [0.9 0.8], [0.9 0.05 0.05], 1) */
void mexFunction( int nlhs, mxArray *plhs[], 
		  int nrhs, const mxArray*prhs[] )
     
{ 
    double *ai, *p_surv_idio, *Y_int; 
    double *loss;
    unsigned int num_firms; 

    /* Get number of companies, number of points in time, and number of points for numerical integration */
    num_firms = mxGetN(prhs[1]);
    
    /* Create a matrix for the return argument */ 
    plhs[0] = mxCreateDoubleMatrix(num_firms+1, 1, mxREAL); 
    
    /* Assign pointers to the various parameters */ 
    loss = mxGetPr(plhs[0]);
    ai = mxGetPr(prhs[0]);
    p_surv_idio = mxGetPr(prhs[1]);
    Y_int = mxGetPr(prhs[2]);
    
    /* Do the actual computations in a subroutine */
    cond_loss_dist_wrapper(loss, ai, p_surv_idio, Y_int, num_firms);
    return;
}